/*
 * $Id: window.c,v 1.16 2005/08/16 20:57:45 baum Exp $
 *
 * This file implements the window widget
 *
 * Copyright (c) 2001 - 2005 Peter G. Baum  http://www.dr-baum.net
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

/****h* widget/window
 * NAME
 * 	window -- A standard top level window.
 *
 * AUTHOR
 * 	Peter G. Baum
 * 	William J Giddings
 *
 * CREATION DATE
 * 	2001-03
 *
 * PURPOSE
 *
 * USAGE
 *
 * SYNOPSIS
 *	gnocl::windows [-option value...]
 *
 * OPTIONS
 *	-acceptFocus
 * 		type: boolean
 *		XXXXX
 *
 *	-allowGrow
 *		type: boolean
 *		Whether the window may be greater than its children require.
 *
 *	-allowShrink
 * 		type: boolean
 * 		Whether the window can be made smaller than the place that its
 * 		children require.
 *
 *	-borderWidth
 * 		type: integer or one of small, normal or big
 * 		Space in pixel between the children and the border of the widget.
 *
 *	-backgroundColor
 *		type: color
 *		Set the background colour for the text widget.
 *
 *	-backgroundImage
 * 		type: string
 * 		Set the window background to an image.
 *		Note: If the window is resized larger thatn the image, then it
 * 		will be tiles to fit.
 *
 *	-child
 * 		type: widget-ID
 *		Widget ID of the child.
 *
 *	-data
 * 		type: string
 * 		User defined data which can be retrieved via the cget subcommand.
 *
 *	-decorated
 * 		type: boolean
 * 		Whether the window should be decorated or not.
 *
 *	-defaultHeight
 * 		type: integer
 * 		Default height of the window.
 *
 *	-defaultWidth
 *		type: integer
 *		Default width of the window.
 *
 * 	-deletable
 * 		type: boolean
 * 		XXXXX
 *
 * 	-destroyWithParent
 *		type: boolean
 *		XXXXX
 *
 *	-dragTargets
 * 		type: list of strings
 * 		List of source targets (e.g. text/plain or application/x-color)
 * 		which are supported.
 *
 * 	-dropTargets
 *  	type: list of strings
 *		List of destination targets (e.g. text/plain or application/x-color)
 * 		which are supported.
 *
 *	-focusOnMap
 * 		type: boolean
 * 		XXXXX
 *
 *	-gravity
 * 		type: boolean
 * 		XXXXX
 *
 * 	-hasTopLevelFocus  <- make this a command, not an option
 * 		type: boolean
 * 		XXXXX
 *
 *	-height
 *		type: integer
 *		Height of the window.
 *
 *	-heightRequest
 *		type: integer
 *		Requested height of the window.
 *
 *	-icon
 *		type: percent-string (default: "")
 *		Sets the icon of the window which is shown, depending on the
 * 		window manager, in the window decoration, the window list, and/or
 * 		if the window is iconified.
 * 		This must be either a file name (prefix "%/") or empty.
 *
 * 	-isActive
 * 		type: boolean
 * 		XXXXX
 *
 *	-keepAbove
 * 		type: boolean (default: 0)
 * 		Whether to force a window to remain on top of all other windows
 * 		on screen.
 *
 *	-keepBelow
 * 		type: boolean (default: 0)
 *		Whether to force a window to remain beneath all other windows
 * 		on screen.
 *
 *	-mask
 * 		type: string
 * 		Applies a shape mask to the window.
 *
 *	-name
 * 		type: string
 * 		Name of the widget, can be used to set options in an rc file.
 *
 * 	-onDelete
 *		type: string (default: "")
 * 		Tcl command which is executed if the widget shall be deleted.
 * 		If the command returns 0, the widget is not deleted.
 * 		Before evaluation the following percent strings are substituated:
 * 		%w	widget name.
 *
 * 	-onDestroy
 *     	type: string (default: "")
 * 	    Tcl command which is executed if the widget is destroyed.
 * 		Before evaluation the following percent strings are substituated:
 * 			%w	widget name
 *
 *	-onDragData
 *     	type: string (default: "")
 * 		Tcl command which is executed if data is draged from this the
 * 		widget to another.
 * 		This command must return the data to be draged.
 * 		Before evaluation the following percent strings are substituated:
 * 			%w	widget name
 * 			%t	time
 * 			%T	type of data
 *
 * 	-onDropData
 *		type: string (default: "")
 *		Tcl command which is executed if data is dropped on the widget.
 * 		Before evaluation the following percent strings are substituated:
 * 			%w	widget name
 * 			%d	data
 * 			%l	length of data
 * 			%t	time
 * 			%T	type of data
 *			%x	x coordinate
 * 			%y	y coordinate
 *
 * 	-onEnter
 * 		type: string (default: "")
 * 		Tcl script which is executed whenever the mouse pointer enters
 * 		the window.
 *		Before evaluation the following percent strings are substituated:
 * 			%w 	widget name of the parent the handlebox
 * 			%x 	x coordinate
 * 			%y 	y coordinate
 * 			%s 	state of the buttons and modifiers (bitmask)
 *
 * 	-onLeave
 * 		type: string (default: "")
 * 		Tcl script which is executed whenever the mouse pointer leaves
 * 		the window.
 *		Before evaluation the following percent strings are substituated:
 * 			%w 	widget name of the parent the handlebox
 * 			%x 	x coordinate
 * 			%y 	y coordinate
 * 			%s 	state of the buttons and modifiers (bitmask)
 *
 * 	-onKeyPress
 * 		type: string (default: "")
 * 		Tcl command which is executed if a key is pressed while the widget
 * 		is having the focus.
 * 		Before evaluation the following percent strings are substituated:
 * 			%w	widget name
 * 			%k	key code as integer
 * 			%K	key code as symbol
 * 			%a	unicode unicode character, or the empty string if there
 * 				is no corresponding character.
 *  			%s	state of the buttons and modifiers (bitmask)
 *
 *	-onKeyRelease
 *		type: string (default: "")
 * 		Tcl command which is executed if a key is released while the widget
 * 		has the focus.
 * 		Before evaluation the following percent strings are substituated:
 * 			%w	widget name
 * 			%k	key code as integer
 * 			%K	key code as symbol
 * 			%a	unicode unicode character, or the empty string if there
 * 				is no corresponding character.
 * 			%s	state of the buttons and modifiers (bitmask)
 *
 * 	-onPopupMenu
 *		type: string (default: "")
 *		Tcl command which is executed if the "popup-menu" signal is recieved,
 * 		which is normally the case if the user presses Shift-F10.
 * 		Before evaluation the following percent strings are substituated:
 * 			%w	widget name
 *
 *	-opacity
 * 		type: double (defaul: 1.0)
 *		Sets the opacity of the window to a decimal value between 0 (fully
 * 		transparent) and 1 (totally opaque).
 *
 *	-onRealize
 *    	type: string (default: "")
 *	    Tcl command whih is executed in the global scope if the widget
 * 		has been realized.
 * 		Before evaluation the following percent strings are substituated:
 * 		%w	widget name.
 *
 *	-onShowHelp
 *		type: string (default: "")
 *		Tcl command which is executed in the global scope if the "show-help"
 * 		signal is recieved, which is normally the case if the user presses
 * 		F1 or Ctrl-F1.
 * 		Before evaluation the following percent strings are substituated
 * 			%w	widget name
 * 			%h	help type: either "whatsThis" or "tooltip"
 *
 * 	-resizable
 *		type: boolean (default: 1)
 *		Whether the user can change the size of the window.
 *
 *	-sensitive
 *		type: boolean (default: 1)
 *		Whether or not the item is sensitve to user input.
 *
 *	-stick
 *		type: boolean (default: 0)
 *		Whether or not the window is to appear on all user desktops.
 *
 *	-title
 *		type: string (default: "")
 *		Title of the window.
 *
 *	-tooltip
 *		type: string
 *		Message that appear next to this widget when the mouse pointer
 * 		is held over it for a short amount of time.
 *
 *	-transient
 *		type: string
 * 		Name of the widget for which the newly created window will become
 * 		transient.
 *		Note: This setting is used to create floating palettes. To prevent palette re-sizing,
 * 		use the set the -allowGrow option to FALSE.
 *
 *	-modal
 *		type: boolean (default: 0)
 *		Whether the window is modal, i.e. it grabs all GTK+ events.
 *
 *	-visible
 *		type: boolean (default: 1)
 *		Whether or not the item is visible.
 *
 *	-width
 *		type: integer
 *		Width of the window.
 *
 *	-widthRequest
 *		type: integer
 *		Requested width of the window.
 *
 *	-x
 *		type: integer
 *		X position of the window.
 *
 * 	-y
 *		type: integer
 *		Y position of the window.
 *
 *	Description
 *		To be visible each widget must be a direct or indirect children
 * 		of a top level window. The window widget is the standard top level
 * 		window. Its appearance is dependent of the window manager.
 *
 * COMMANDS
 *
 *	id cget option
 *		Returns the value for one option. The option may have any of the
 * 		values accepted by configure.
 *
 *	id configure [-option value...]
 *		Configures the widget. Option may have any of the values accepted
 * 		on creation of the widget.
 *
 *	id delete
 *		Deletes the widget and the associated tcl command.
 *
 *	id iconify ?state?
 *		Iconifies or deiconifies the window dependent on the value of state.
 * 		The default value for state is true.
 *
 *	id reshow
 * 		Reset and then redisplay the window using its default position
 * 		and size.
 *
 *	id geometry
 * 		Returns a list containing the current position and size of the
 * 		window as root_x, root_y, width and height.
 *
 * 	id pointer
 * 		Returns a list containing the current position of the pointer
 * 		within the window as x and y.
 *
 *	id reposition
 * 		Resize and relocate the window using geometry list as root_x,
 * 		root_y, width and height.
 *
 *	See also
 *		dialog, GtkWindow
 *
 * MODIFICATION HISTORY
 *  2009-02: added -bakckgroundImage, -mask
 *	2009-01: added -keepAbove, -keepBelow, -acceptFocus,
 *			-deletable, -destroyWithParent, -focusOnMap,
 *			-gravity, -hasTopLevelFocus, -isActive,
 *			-skipPagerHint, -skipTaskBarHint, -urgencyHint,
 *			-backgroundColor, fullScreen, geometry
 *	2008-10:	added -transient, class
 *	2008-03: new windows are always centered on screen
 *	2007-10: added center
 *	2004-02: added -data
 *		 11: added cget, -x, -y, -width and -height
 *		 10: added onKey{Press,Release}
 *		 04: added -icon
 *		 02: added drag and drop options
 * 	2003-01: small fixes, added options "-name"
 *		 09: switched from GnoclWidgetOptions to GnoclOption
 *			 more updates, more and some renamed options
 *	2002-04: update for gtk 2.0
 *	2001-03: Begin of developement
 *
 * TODO
 * 	options
 * 	//gtk_window_maximize(GTK_WINDOW(window));

 *		add_accel_group
 *		remove_accel_group
 * 		set_modal
 * 		set_default_size
 * 		set_geometry_hints
 * 		set_gravity
 * 		get_gravity
 * 		set_position
 * 		set_screen
 * 		get_screen
 * 		is_active
 * 		get_focus
 * 		set_focus
 * 		set_default
 * 		set_frame_dimensions
 * 		set_has_frame
 * 		set_role
 * 		get_role
 *		set_startup_id
 * 		set_default_icon_list
 * 		default_icon_list
 *		get_icon_list
 *		get_icon_name
 * 		set_default_icon_name
 * 		set_icon
 * 		set_icon_list
 *		set_icon_from_file
 * 		set_icon_name
 * 		set_auto_startup_notification
 * 	commands
 *		activate_focus
 * 		activate_default
 * 		list_toplevels
 * 		add_mnemonic
 * 		remove_mnemonic
 * 		set_mnemonic_modifier
 *		get_mnemonic_modifier
 * 		mnemonic_activate
 * 		activate_key
 * 		propagate_key_event
 * 		present
 * 		present_with_time
 * 		begin_resize_drag
 * 		begin_move_drag
 * 		get_modal
 * 		window_get_position
 * 		get_size
 * 		get_group
 * 		move
 * 		parse_geometry
 * 		resize
 * Signals
 *		activate-default
 * 		activate-focus
 * 		frame-event
 *		keys-changed
 * 		set-focus
 * BUGS
 * EXAMPLE
 * |html <B></t>PRODUCES</B><Br><Br>
 * |html <image src="../pics/window.png">
 * SEE ALSO
 * SOURCE
 *****/

#include "gnocl.h"
#include <gdk/gdk.h>	/* wjg 05/07/08 */
#include <string.h>
#include <assert.h>
#include <stdlib.h>		/* atof */

/*****/



static GdkBitmap *get_bitmap ( gchar *filename )
{
	GdkPixbuf *pbuf;
	GdkBitmap *ret;
	GError *err = NULL;

	g_return_val_if_fail ( filename != NULL, NULL );

	pbuf = gdk_pixbuf_new_from_file ( filename, &err );

	if ( err != NULL )
	{
		g_warning ( "%s", err->message );
		g_error_free ( err );
		return NULL;
	}

	/* you may want to change the threshold, depending on your image */
	gdk_pixbuf_render_pixmap_and_mask ( pbuf, NULL, &ret, 1 );

	g_object_unref ( pbuf );

	return ret;
}


/****v* window/windowOptions
 * AUTHOR
 *	Peter G. Baum
 * SOURCE
 */
static GnoclOption windowOptions[] =
{
	{ "-defaultWidth", GNOCL_INT, "default-width" },     /* these must be */
	{ "-defaultHeight", GNOCL_INT, "default-height" },   /* before -visible! */
	{ "-decorated", GNOCL_BOOL, "decorated" },           /* 2 */
	{ "-visible", GNOCL_BOOL, "visible" },               /* 3 */
	{ "-x", GNOCL_INT, NULL },                           /* 4 */
	{ "-y", GNOCL_INT, NULL },                           /* 5 */
	{ "-width", GNOCL_INT, NULL },                       /* 6 */
	{ "-height", GNOCL_INT, NULL },                      /* 7 */
	{ "-allowGrow", GNOCL_BOOL, "allow-grow" },
	{ "-allowShrink", GNOCL_BOOL, "allow-shrink" },
	{ "-borderWidth", GNOCL_OBJ, "border-width", gnoclOptPadding },
	{ "-child", GNOCL_OBJ, "", gnoclOptChild },
	{ "-data", GNOCL_OBJ, "", gnoclOptData },
	{ "-dragTargets", GNOCL_LIST, "s", gnoclOptDnDTargets },
	{ "-dropTargets", GNOCL_LIST, "t", gnoclOptDnDTargets },
	{ "-heightRequest", GNOCL_INT, "height-request" },
	{ "-icon", GNOCL_OBJ, "", gnoclOptIcon },
	{ "-modal", GNOCL_BOOL, "modal" },
	{ "-name", GNOCL_STRING, "name" },
	{ "-onDelete", GNOCL_OBJ, "", gnoclOptOnDelete },
	{ "-onDestroy", GNOCL_OBJ, "destroy", gnoclOptCommand },
	{ "-onDragData", GNOCL_OBJ, "", gnoclOptOnDragData },
	{ "-onDropData", GNOCL_OBJ, "", gnoclOptOnDropData },
	{ "-onKeyPress", GNOCL_OBJ, "", gnoclOptOnKeyPress },
	{ "-onKeyRelease", GNOCL_OBJ, "", gnoclOptOnKeyRelease },
	{ "-onPopupMenu", GNOCL_OBJ, "popup-menu", gnoclOptCommand },
	{ "-onRealize", GNOCL_OBJ, "realize", gnoclOptCommand },
	{ "-onResize", GNOCL_OBJ, "", gnoclOptOnConfigure },
	{ "-onShowHelp", GNOCL_OBJ, "", gnoclOptOnShowHelp },
	{ "-resizable", GNOCL_BOOL, "resizable" },
	{ "-sensitive", GNOCL_BOOL, "sensitive" },
	{ "-title", GNOCL_STRING, "title" },
	{ "-widthRequest", GNOCL_INT, "width-request" },
	{ "-typeHint", GNOCL_OBJ, "", gnoclOptWindowTypeHint },
	{ "-tooltip", GNOCL_OBJ, "", gnoclOptTooltip },

	{ "-transient", GNOCL_OBJ, "", gnoclOptTransientWindow },
	{ "-keepAbove", GNOCL_OBJ, "", gnoclOptKeepAbove },
	{ "-keepBelow", GNOCL_OBJ, "", gnoclOptKeepBelow },
	{ "-stick", GNOCL_OBJ, "", gnoclOptStick },
	{ "-opacity", GNOCL_OBJ, "", gnoclOptOpacity },

	{ "-acceptFocus", GNOCL_BOOL, "accept-focus" },
	{ "-deletable", GNOCL_BOOL, "deletable" },
	{ "-destroyWithParent", GNOCL_BOOL, "destroy-with-parent" },
	{ "-focusOnMap", GNOCL_BOOL, "focus-on-map" },
	{ "-gravity", GNOCL_BOOL, "gravity" },
	{ "-hasToplevelFocus", GNOCL_BOOL, "has-toplevel-focus" },
	{ "-isActive", GNOCL_BOOL, "is-active" },
	{ "-fullScreen", GNOCL_OBJ, "", gnoclOptFullScreen},

	{ "-skipPagerHint", GNOCL_BOOL, "skip-pager-hint" },
	{ "-skipTaskBarHint", GNOCL_BOOL, "skip-taskbar-hint" },
	{ "-urgencyHint", GNOCL_BOOL, "urgency-hint" },

	{ "-backgroundColor", GNOCL_OBJ, "normal", gnoclOptGdkColorBg },
	{ "-onEnter", GNOCL_OBJ, "E", gnoclOptOnEnterLeave },
	{ "-onLeave", GNOCL_OBJ, "L", gnoclOptOnEnterLeave },

	{ "-mask", GNOCL_OBJ, "", gnoclOptMask },
	{ "-backgroundImage", GNOCL_OBJ, "", gnoclOptBackgroundImage },
	{ "-cursor", GNOCL_OBJ, "", gnoclOptCursor },
	{ "-hasFrame", GNOCL_OBJ, "", gnoclOptHasFrame },

	{ NULL }
};
/*****/

/****d* window/constants
 * AUTHOR
 *	Peter G. Baum
 * SOURCE
 */
static const int decoratedIdx   = 2;
static const int visibleIdx     = 3;
static const int xIdx           = 4;
static const int yIdx           = 5;
static const int widthIdx       = 6;
static const int heightIdx      = 7;
/*****/

/****f* window/staticFuncs/doOnConfigure
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 *	William J Giddings
 * CREATION DATE
 * 	When created
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
static gboolean doOnConfigure ( GtkWidget *widget, GdkEventConfigure *event, gpointer data )
{
	GnoclCommandData *cs = ( GnoclCommandData * ) data;

	GnoclPercSubst ps[] =
	{
		{ 'w', GNOCL_STRING },  /* widget */
		{ 'x', GNOCL_INT },     /*  */
		{ 'y', GNOCL_INT },     /*  */
		{ 'W', GNOCL_INT },     /*  */
		{ 'H', GNOCL_INT },     /*  */
		{ 0 }
	};

	ps[0].val.str = gnoclGetNameFromWidget ( widget );
	ps[1].val.i = event->x;
	ps[2].val.i = event->y;
	ps[3].val.i = event->width;
	ps[4].val.i = event->height;
	gnoclPercentSubstAndEval ( cs->interp, ps, cs->command, 1 );

	return FALSE;
}

/*****/

/****f* window/gnoclOptOnConfigure
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 * 	Peter G. Baum
 * 	William J Giddings
 * CREATION DATE
 * 	When created
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclOptOnConfigure ( Tcl_Interp *interp, GnoclOption *opt, GObject *obj, Tcl_Obj **ret )
{
	assert ( strcmp ( opt->optName, "-onResize" ) == 0 );
	return gnoclConnectOptCmd ( interp, obj, "configure-event", G_CALLBACK ( doOnConfigure ), opt, NULL, ret );
}

/*****/

/****f* window/gnoclOptTransientWindow
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 * 	William J Giddings
 * CREATION DATE
 * 	26-10-08
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclOptTransientWindow (
	Tcl_Interp	*interp,
	GnoclOption	*opt,
	GObject		*obj,
	Tcl_Obj		**ret )
{

	/* get the name of the parent window from options */
	const char *name = Tcl_GetString ( opt->val.obj );
	GtkWidget *parent = gnoclGetWidgetFromName ( name, interp );

	gtk_window_set_transient_for (  GTK_WINDOW ( obj ) , GTK_WINDOW ( parent ) );
	gtk_window_set_destroy_with_parent  ( GTK_WINDOW ( obj ) , 1 );

	return TCL_OK;
}

/*****/

/****f* window/gnoclOptKeepAbove
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 *	William J Giddings
 * CREATION DATE
 * 	12-01-09
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 *	Causes window to remain displayed on top of all others.
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */

int gnoclOptKeepAbove (
	Tcl_Interp	*interp,
	GnoclOption	*opt,
	GObject		*obj,
	Tcl_Obj		**ret )
{

	if ( strcmp ( Tcl_GetString ( opt->val.obj ), "1" ) == 0 )
	{
		gtk_window_set_keep_above ( GTK_WINDOW ( obj ), 1 );
	}

	else
	{
		gtk_window_set_keep_above ( GTK_WINDOW ( obj ), 0 );
	}

	return TCL_OK;
}

/*****/

/****f* window/gnoclOptKeepBelow
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 *	William J Giddings
 * CREATION DATE
 * 	12-01-09
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 *	Causes window to remain displayed below of all others.
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */

int gnoclOptKeepBelow (
	Tcl_Interp	*interp,
	GnoclOption	*opt,
	GObject		*obj,
	Tcl_Obj		**ret )
{

	if ( strcmp ( Tcl_GetString ( opt->val.obj ), "1" ) == 0 )
	{
		gtk_window_set_keep_below ( GTK_WINDOW ( obj ), 1 );
	}

	else
	{
		gtk_window_set_keep_below ( GTK_WINDOW ( obj ), 0 );
	}

	return TCL_OK;
}

/*****/

/****f* window/gnoclOptFullScreen
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 *	William J Giddings
 * CREATION DATE
 * 	12-01-09
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 *	Causes window to remain displayed on top of all others.
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */

int gnoclOptFullScreen (
	Tcl_Interp	*interp,
	GnoclOption	*opt,
	GObject		*obj,
	Tcl_Obj		**ret )
{

	if ( strcmp ( Tcl_GetString ( opt->val.obj ), "1" ) == 0 )
	{
		gtk_window_fullscreen ( GTK_WINDOW ( obj ) );
	}

	else
	{
		gtk_window_set_keep_above ( GTK_WINDOW ( obj ), 0 );
		gtk_window_unfullscreen ( GTK_WINDOW ( obj ) );
	}

	return TCL_OK;
}

/*****/

/****f* window/gnoclOptStick
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 * 	Peter G. Baum
 * 	William J Giddings
 * CREATION DATE
 * 	12-01-09
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * 	Causes window to be displayed on all desktops.
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclOptStick (
	Tcl_Interp	*interp,
	GnoclOption	*opt,
	GObject		*obj,
	Tcl_Obj		**ret )
{

	if ( strcmp ( Tcl_GetString ( opt->val.obj ), "1" ) == 0 )
	{
		gtk_window_stick   ( GTK_WINDOW ( obj ) );
	}

	else
	{
		gtk_window_unstick ( GTK_WINDOW ( obj ) );
	}

	return TCL_OK;
}

/*****/

/****f* window/gnoclOptMask
 * NAME
 *	functionName
 * PURPOSE
 * 	Apply mask to the
 * AUTHOR
 * 	William J Giddings
 * CREATION DATE
 * 	12-01-09
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * 	http://www.gnu.org/software/guile-gnome/docs/gdk/html/Windows.html
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclOptMask (
	Tcl_Interp	*interp,
	GnoclOption	*opt,
	GObject		*obj,
	Tcl_Obj		**ret )
{

	g_print ( "window/gnoclOptMask\n" );

	GdkBitmap *mask;

	mask = get_bitmap ( Tcl_GetString ( opt->val.obj ) );

	gtk_widget_shape_combine_mask ( obj, mask, 0, 0 );

	return TCL_OK;
}

/*****/


/****f* window/gnoclOptOpacity
 * NAME
 *	functionName
 * PURPOSE
 * 	Apply mask to the
 * AUTHOR
 * 	William J Giddings
 * CREATION DATE
 * 	12-01-09
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * 	http://www.gnu.org/software/guile-gnome/docs/gdk/html/Windows.html
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclOptOpacity (
	Tcl_Interp	*interp,
	GnoclOption	*opt,
	GObject		*obj,
	Tcl_Obj		**ret )
{
	g_print ( "window/gnoclOptOpacity\n" );
	gtk_window_set_opacity ( GTK_WINDOW ( obj ), atof ( Tcl_GetString ( opt->val.obj ) ) );

	return TCL_OK;
}

/*****/

/****f* window/gnoclOptHasFrame
 * NAME
 *	functionName
 * PURPOSE
 *
 * AUTHOR
 * 	William J Giddings
 * CREATION DATE
 * 	09-02-09
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 *
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclOptHasFrame (
	Tcl_Interp	*interp,
	GnoclOption	*opt,
	GObject		*obj,
	Tcl_Obj		**ret )
{
	g_print ( "window/gnoclOptHasFrame\n" );

	gtk_window_set_has_frame ( GTK_WINDOW ( obj ),  Tcl_GetString ( opt->val.obj ) );
	//gtk_window_set_opacity ( GTK_WINDOW ( obj ), atof ( Tcl_GetString ( opt->val.obj ) ) );

	return TCL_OK;
}

/*****/

/****f* window/gnoclOptBackgroundImage
 * NAME
 *	functionName
 * PURPOSE
 * 	Set the opacity of the specified window.
 * AUTHOR
 * 	William J Giddings
 * CREATION DATE
 * 	12-01-09
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclOptBackgroundImage (
	Tcl_Interp	*interp,
	GnoclOption	*opt,
	GObject		*obj,
	Tcl_Obj		**ret )
{

	GdkPixmap *background;
	GdkBitmap *mask;
	GdkPixbuf *pixbuf;
	GdkScreen *ourscreen;
	GdkColormap *colormap;
	GtkStyle *style;
	//GdkColor fg;
	//GdkColor bg;
	GError *error = NULL;

	/* create a pixbuf from a disk file */

	//mask = get_bitmap( Tcl_GetString ( opt->val.obj ) );

	pixbuf = gdk_pixbuf_new_from_file ( Tcl_GetString ( opt->val.obj ), &error );

	if ( error != NULL )
	{
		if ( error->domain == GDK_PIXBUF_ERROR )
		{
			g_print ( "Pixbuf Related Error:\n" );
		}

		if ( error->domain == G_FILE_ERROR )
		{
			g_print ( "File Error: Check file permissions and state:\n" );
		}

		g_printerr ( "%s\n", error[0].message );

		exit ( 1 );
	}

	//gdk_pixbuf_render_pixmap_and_mask ( pixbuf, &background, &mask, 0 );
	gdk_pixbuf_render_pixmap_and_mask ( pixbuf, &background, NULL, 0 );

	style = GTK_WIDGET ( obj )->style;

	style->bg_pixmap[0] = background;

	gtk_widget_set_style ( GTK_WIDGET ( obj ), GTK_STYLE ( style ) );

	return TCL_OK;

}

/* just a dummy */
int gnoclLabelEntryCmd (
	ClientData data,
	Tcl_Interp *interp,
	int objc,
	Tcl_Obj * const objv[] )
{
	return TCL_OK;
}

/*****/



/****f* window/gnoclOptWindowTypeHint
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 * 	Peter G. Baum
 * 	William J Giddings
 * CREATION DATE
 * 	When created
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclOptWindowTypeHint ( Tcl_Interp *interp, GnoclOption *opt, GObject *obj, Tcl_Obj **ret )
{
	const char *txt[] = { "normal", "dialog", "menu", "toolbar", "splashscreen", "utility", "dock", "desktop", NULL
						};
	GdkWindowTypeHint types[] = { GDK_WINDOW_TYPE_HINT_NORMAL,
								  GDK_WINDOW_TYPE_HINT_DIALOG, GDK_WINDOW_TYPE_HINT_MENU,
								  GDK_WINDOW_TYPE_HINT_TOOLBAR, GDK_WINDOW_TYPE_HINT_SPLASHSCREEN,
								  GDK_WINDOW_TYPE_HINT_UTILITY, GDK_WINDOW_TYPE_HINT_DOCK,
								  GDK_WINDOW_TYPE_HINT_DESKTOP
								};

	if ( ret == NULL ) /* set value */
	{
		int idx;

		if ( Tcl_GetIndexFromObj ( interp, opt->val.obj, txt, "type hint", TCL_EXACT, &idx ) != TCL_OK )
			return TCL_ERROR;

		gtk_window_set_type_hint ( GTK_WINDOW ( obj ), types[idx] );
	}

	else /* get value */
	{
		GdkWindowTypeHint val = gtk_window_get_type_hint ( GTK_WINDOW ( obj ) );
		int k;

		for ( k = 0; txt[k]; ++k )
		{
			if ( types[k] == val )
			{
				*ret = Tcl_NewStringObj ( txt[k], -1 );
				return TCL_OK;
			}
		}

		Tcl_SetResult ( interp, "Unknown setting for parameter", TCL_STATIC );

		return TCL_ERROR;
	}

	return TCL_OK;
}

/*****/

/****f* window/staticFuncs/configure
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 *	William J Giddings
 * CREATION DATE
 *	When created
 * USAGE
 *	How this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */

static int configure ( Tcl_Interp *interp, GtkWindow *window, GnoclOption options[] )
{

	/* make only one move if x and y are set */
	if ( options[xIdx].status == GNOCL_STATUS_CHANGED
			&& options[xIdx].status == GNOCL_STATUS_CHANGED )
	{
		gtk_window_move ( window, options[xIdx].val.i, options[yIdx].val.i );
	}

	else if ( options[xIdx].status == GNOCL_STATUS_CHANGED
			  || options[yIdx].status == GNOCL_STATUS_CHANGED )
	{
		int x, y;
		gtk_window_get_position ( window, &x, &y );

		if ( options[xIdx].status == GNOCL_STATUS_CHANGED )
			x = options[xIdx].val.i;
		else
			y = options[yIdx].val.i;

		gtk_window_move ( window, x, y );
	}

	/* get_size does not return size set by resize if the resize event
	   is not yet handled, we therefor use get_size only if really necessary */

	if ( options[widthIdx].status == GNOCL_STATUS_CHANGED
			&& options[heightIdx].status == GNOCL_STATUS_CHANGED )
	{
		gtk_window_resize ( window, options[widthIdx].val.i,
							options[heightIdx].val.i );
	}

	else if ( options[widthIdx].status == GNOCL_STATUS_CHANGED
			  || options[heightIdx].status == GNOCL_STATUS_CHANGED )
	{
		int width, height;
		gtk_window_get_size ( window, &width, &height );

		if ( options[widthIdx].status == GNOCL_STATUS_CHANGED )
			width = options[widthIdx].val.i;
		else
			height = options[heightIdx].val.i;

		gtk_window_resize ( window, width, height );
	}

	return TCL_OK;
}

/*****/

/****f* window/staticFuncs/cget
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 *	William J Giddings
 * CREATION DATE
 *	When created
 * USAGE
 *	How this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */

static int cget ( Tcl_Interp *interp, GtkWindow *window,
				  GnoclOption options[], int idx )
{
	Tcl_Obj *obj = NULL;

	if ( idx == xIdx )
	{
		int x, y;
		gtk_window_get_position ( window, &x, &y );
		obj = Tcl_NewIntObj ( x );
	}

	else if ( idx == yIdx )
	{
		int x, y;
		gtk_window_get_position ( window, &x, &y );
		obj = Tcl_NewIntObj ( y );
	}

	else if ( idx == widthIdx )
	{
		int width, height;
		gtk_window_get_size ( window, &width, &height );
		obj = Tcl_NewIntObj ( width );
	}

	else if ( idx == heightIdx )
	{
		int width, height;
		gtk_window_get_size ( window, &width, &height );
		obj = Tcl_NewIntObj ( height );
	}

	if ( obj != NULL )
	{
		Tcl_SetObjResult ( interp, obj );
		return TCL_OK;
	}

	return gnoclCgetNotImplemented ( interp, options + idx );
}

/*****/

/****f* window/staticFuncs/windowFunc
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 * 	Peter G. Baum
 * 	William J Giddings
 * CREATION DATE
 * 	When created
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 *  objv[0] = widget name
 *  objv[1] = command
 *  objv[2] = paramter1
 *  objv[3] = paramter2
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
static int windowFunc ( ClientData data, Tcl_Interp *interp, int objc, Tcl_Obj * const objv[] )
{
	static const char *cmds[] = { "delete", "configure", "cget", "iconify", "center",
								  "beep", "class", "reshow", "geometry", "pointer", "reposition", "grab", "ungrab", NULL
								};
	enum cmdIdx { DeleteIdx, ConfigureIdx, CgetIdx, IconifyIdx, CenterIdx,
				  BeepIdx, ClassIdx, ReshowIdx, GeometryIdx, PointerIdx, RepositionIdx, GrabIdx, UngrabIdx
				};
	GtkWindow *window = GTK_WINDOW ( data );
	int idx;

	if ( objc < 2 )
	{
		Tcl_WrongNumArgs ( interp, 1, objv, "command" );
		return TCL_ERROR;
	}

	if ( Tcl_GetIndexFromObj ( interp, objv[1], cmds, "command",
							   TCL_EXACT, &idx ) != TCL_OK )
		return TCL_ERROR;

	switch ( idx )
	{
			/* WJG added 16/01/09 */
		case GrabIdx:
			{
				gtk_grab_add ( window );
			}

			break;
		case UngrabIdx:
			{
				gtk_grab_remove ( window );
			}

			break;
		case RepositionIdx:
			{
				gint x, y, w, h;

				/* get values from the args list */
				sscanf ( Tcl_GetString ( objv[2] ), "%d %d %d %d", &x, &y, &w, &h );
				gtk_window_move ( window, x, y );
				gtk_window_resize ( window, w, h );
			}

			break;
		case PointerIdx:
			{
				char str[250];
				gint x;
				gint y;
				gtk_widget_get_pointer ( window, &x, &y );
				sprintf ( &str, "%d %d", x, y );
				Tcl_SetObjResult ( interp, Tcl_NewStringObj ( str, -1 ) );
			}

			break;
		case GeometryIdx:
			{
				/* is this task best achieved using gtk or gdk library calls? */
				char str[250];
				gint root_x;
				gint root_y;
				gint width;
				gint height;
				gtk_window_get_position ( window, &root_x, &root_y );
				gtk_window_get_size ( window, &width, &height );
				sprintf ( &str, "%d %d %d %d", root_x, root_y, width, height );
				Tcl_SetObjResult ( interp, Tcl_NewStringObj ( str, -1 ) );
			}

			break;
		case ReshowIdx:
			{
				gtk_window_reshow_with_initial_size ( window );
			}

			break;
		case ClassIdx:
			{
				Tcl_SetObjResult ( interp, Tcl_NewStringObj ( "window", -1 ) );
			}

			break;
		case DeleteIdx:
			{
				return gnoclDelete ( interp, GTK_WIDGET ( window ), objc, objv );
			}

			break;
		case ConfigureIdx:
			{
				int ret = TCL_ERROR;

				if ( gnoclParseAndSetOptions ( interp, objc - 1, objv + 1,
											   windowOptions, G_OBJECT ( window ) ) == TCL_OK )
				{
					ret = configure ( interp, window, windowOptions );
				}

				gnoclClearOptions ( windowOptions );

				return ret;
			}

			break;
		case CgetIdx:
			{
				int     idx;

				switch ( gnoclCget ( interp, objc, objv, G_OBJECT ( window ),
									 windowOptions, &idx ) )
				{
					case GNOCL_CGET_ERROR:
						return TCL_ERROR;
					case GNOCL_CGET_HANDLED:
						return TCL_OK;
					case GNOCL_CGET_NOTHANDLED:
						return cget ( interp, window, windowOptions, idx );
				}

				assert ( 0 );
			}

			break;
		case IconifyIdx:
			{
				int iconify = 1;

				if ( objc == 3 )
				{
					if ( Tcl_GetBooleanFromObj ( interp, objv[2], &iconify )
							!= TCL_OK )
						return TCL_ERROR;
				}

				else if ( objc > 3 )
				{
					Tcl_WrongNumArgs ( interp, 2, objv, "?state?" );
					return TCL_ERROR;
				}

				if ( iconify )
					gtk_window_iconify ( window );
				else
					gtk_window_deiconify ( window );
			}

			break;
		case CenterIdx:
			{
				/* WJG (28/03/08)  For some reason GTK_WIN_POS_CENTER has no effect! */
				gtk_window_set_position ( window , GTK_WIN_POS_CENTER_ALWAYS );
			}

			break;
	}

	return TCL_OK;
}

/*****/

/****f* window/gnoclWindowCmd
 * NAME
 *	functionName
 * PURPOSE
 * AUTHOR
 * 	Peter G. Baum
 * 	William J Giddings
 * CREATION DATE
 * 	When created
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTE
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclWindowCmd (
	ClientData data,
	Tcl_Interp *interp,
	int objc,
	Tcl_Obj * const objv[] )
{
	int        ret;
	GtkWindow  *window;

	assert ( strcmp ( windowOptions[visibleIdx].optName, "-visible" ) == 0 );

	if ( gnoclParseOptions ( interp, objc, objv, windowOptions ) != TCL_OK )
	{
		gnoclClearOptions ( windowOptions );
		return TCL_ERROR;
	}

	window = GTK_WINDOW ( gtk_window_new ( GTK_WINDOW_TOPLEVEL ) );

	/* WJG (28/03/08) New windows are always centred on screen */
	gtk_window_set_position ( window , GTK_WIN_POS_CENTER );

	/* FIXME: each window own accel_group */
	gtk_window_add_accel_group ( window, gnoclGetAccelGroup() );

	ret = gnoclSetOptions ( interp, windowOptions, G_OBJECT ( window ), -1 );

	if ( ret == TCL_OK )
	{
		/* This must be after setting "default-{width,height}".
		   If it is after setting the child widget, we get problems if
		   the child contains a combo widget. Bizarre!
		*/
		if ( windowOptions[visibleIdx].status == 0 )
		{
			gtk_widget_show ( GTK_WIDGET ( window ) );
		}

		ret = configure ( interp, window, windowOptions );
	}

	gnoclClearOptions ( windowOptions );

	if ( ret != TCL_OK )
	{
		gtk_widget_destroy ( GTK_WIDGET ( window ) );
		return TCL_ERROR;
	}

	return gnoclRegisterWidget ( interp, GTK_WIDGET ( window ), windowFunc );
}

/*****/
